  /*
   *  Object %name    : %
   *  State           :  %state%
   *  Creation date   :  Mon Apr 01 12:09:38 2009
   *  Last modified   :  %modify_time%
   */
  /** @file
   *  \brief A brief description of this module
   *
   *  \version SST_SEP_ATP_Authenticators.c#1:csrc:1
   *  \author taniam
   *  \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
   *           All Rights reserved
   */
/************* Include Files ***********************************/
#include "stdio.h"
#include "SST_SEP_TST.h"
#include "DX_VOS_BaseTypes.h"
#include "extapp_err.h"
#include "sep_utils.h"
#include "SEP_ATP_UTIL_funcs.h"

/************************ Global Data **************************/
extern TST_ATP_PassAuthTestDataStruct* TST_ATP_PassAuthObjectTestData;
extern TST_ATP_RsaAuthTestDataStruct* TST_ATP_RsaAuthObjectTestData;

/*****************************************************************************
 * Function Name:
 *  SEPQA_SST_PasswordAuthenticator
 * 
 * Inputs:
 *  None
 *
 * Outputs:
 *  DxError_t - Function error return
 *
 * Description:
 * Test the creation, deletion, modification, opening, 
 * closing and usage of a password authenticator object. 
 * And demonstrate the use of transaction with authenticator objects
 *
 * Algorithm:
 *    1.  Create a new session by calling: SST_SessionCreate.
 *    2.  Open a transaction by calling: SST_TransactionStart.
 *    3.  Create a new master password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    4.  Close the transaction by calling: SST_TransactionEnd
 *    5.  Open the master password authentication by calling: SST_AuthenticationOpen.
 *    6.  Open a transaction by calling: SST_TransactionStart.
 *    7.  Create a new secondary password authenticator by calling: SST_AuthenticatorPasswordCreate.
 *    8.  Close the transaction by calling: SST_TransactionEnd. 
 *    9.  Open the secondary authenticator by calling: SST_AuthenticationOpen.
 *    10. Open a transaction by calling: SST_TransactionStart.
 *    11. Modify the password for the authenticator by calling: SST_AuthenticatorPasswordResetAndChange.
 *    12. Close the authenticator by calling: SST_AuthenticationClose.
 *    13. Close the transaction by calling: SST_TransactionEnd.
 *    14. Reopen the secondary authenticator using the new password by calling: SST_AuthenticationOpen.
 *    15. Close the authenticator by calling: SST_AuthenticationClose.
 *    16. Open the master authenticator using the new password by calling: SST_AuthenticationOpen.
 *    17. Open a transaction by calling: SST_TransactionStart.
 *    18. Insert a new data object by calling: SST_InsertData.
 *    19. Delete the data by calling: SST_DeleteData.
 *    20. Close the authenticator by calling: SST_AuthenticationClose.
 *    21. Delete the authenticators (one by one) by calling: SST_AuthenticatorDelete.
 *    22. Close the transaction by calling: SST_TransactionClose.
 *    23. Delete the session by calling: SST_SessionDelete.
 ********************************************************************************/
DxError_t SEPQA_SST_PasswordAuthenticator ()
{
    /********* TST Variables Definitions ************/
    DxUint32_t                  TST_index = 0;
    DxChar                      TST_MasterTestName[TST_FUNC_STRING_LEN] = "SST_TST_ATP_PasswordAuthenticator";
    DxUint32_t                  TST_Status = 0;
    DxUint32_t                  TST_TestStatus = 0;
    SSTSessionId_t              TST_SessionId = 0; 
    SSTTxnId_t                  TST_TransactionId = 0;
    SSTHandle_t                 TST_MasterPasswordAuthHandle;
    SSTHandle_t                 TST_SecondaryPasswordAuthHandle;
    SSTHandle_t                 TST_DataHandle;
    SSTDataType_t               TST_SimpleData = TST_DATA_OBJECT;
    SSTHandle_t                 TST_AuthDefultHandle;

    /********* TST Variables initialization *********/
    TST_TestStatus = TESTPASS;
    TST_MasterPasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE; 
    TST_SecondaryPasswordAuthHandle.objId = SST_HANDLE_RANDOM_GENERATE;
    TST_DataHandle.objId = SST_HANDLE_RANDOM_GENERATE;

    /************************************************/
    /*********** Start of test **********************/
    /************************************************/
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ******** SEPQA_SST_ATP_PasswordAuthenticator Test  *************** \n"));     
    ATP_LOG_TST_PRINT((SST_TST_MSG," ****************************************************************** \n \n"));

    ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," TEST using internal handles      \n"));
    ATP_LOG_TST_PRINT((SST_TST_MSG," ************************************ \n"));

    /* Create a session for operations */
    /*---------------------------------*/
    TST_Status = SST_SessionCreate(&TST_SessionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionCreate",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open a transaction for operations */
    /*-----------------------------------*/
    TST_Status = SST_TransactionStart(&TST_TransactionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Create a new master password authenticator */
    /*--------------------------------------------*/
    TST_AuthDefultHandle.objDigest = 0;
    TST_AuthDefultHandle.objId = SST_CLOSE_AUTH_HANDLE;

    TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                 TST_SessionId, 
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterMaxNumOfTrials,
                                                 TST_AuthDefultHandle,
                                                 &TST_MasterPasswordAuthHandle,
                                                 TST_Workspace,
                                                 TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate - Master Authenticator",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the transaction */
    /*-----------------------*/
    TST_Status = SST_TransactionEnd(TST_TransactionId,
                                    DX_TRUE,/*reflash sensitive*/
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open the master password authentication */
    /*-----------------------------------------*/

    TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                        TST_MasterPasswordAuthHandle,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                        TST_NO_CHALLENGE,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Master Authentication",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open a transaction for operations */
    /*-----------------------------------*/
    TST_Status = SST_TransactionStart(&TST_TransactionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Create a new password authenticator */
    /*-------------------------------------*/
    TST_Status = SST_AuthenticatorPasswordCreate(TST_TransactionId,
                                                 TST_SessionId,
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPassword,
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPasswordSizeInBytes,
                                                 TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryMaxNumOfTrials,
                                                 TST_MasterPasswordAuthHandle,
                                                 &TST_SecondaryPasswordAuthHandle,
                                                 TST_Workspace,
                                                 TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordCreate - Secondary Authenticator",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the transaction */
    /*-----------------------*/
    TST_Status = SST_TransactionEnd(TST_TransactionId,
                                    DX_TRUE,/*reflash sensitive*/
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open the secondary authentication */
    /*-----------------------------------*/
    TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                        TST_SecondaryPasswordAuthHandle,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPassword,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_SecondaryPasswordSizeInBytes,
                                        TST_NO_CHALLENGE,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open a transaction for operations */
    /*-----------------------------------*/
    TST_Status = SST_TransactionStart(&TST_TransactionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Modify the secondary authenticator */
    /*------------------------------------*/
    TST_Status = SST_AuthenticatorPasswordResetAndChange(TST_TransactionId,
                                                         TST_SessionId,
                                                         TST_SecondaryPasswordAuthHandle,
                                                         TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPasswordSizeInBytes,                    
                                                         TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPassword,
                                                         TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecMaxNumOfTrials,
                                                         TST_Workspace,
                                                         TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorPasswordResetAndChange",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the secondary authentication */
    /*------------------------------------*/
    TST_Status = SST_AuthenticationClose(TST_SessionId,TST_SecondaryPasswordAuthHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the transaction */
    /*-----------------------*/
    TST_Status = SST_TransactionEnd(TST_TransactionId,
                                    DX_TRUE,/*reflash sensitive*/
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Reopen the secondary authenticator using the new password */
    /*-----------------------------------------------------------*/
    TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                        TST_SecondaryPasswordAuthHandle,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPassword,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_NewSecPasswordSizeInBytes,
                                        TST_NO_CHALLENGE,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication new password",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the secondary authentication */
    /*------------------------------------*/
    TST_Status = SST_AuthenticationClose(TST_SessionId,TST_SecondaryPasswordAuthHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose SecondaryPasswordAuthHandle",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open the Master authentication */
    /*--------------------------------*/
    TST_Status = SST_AuthenticationOpen(TST_SessionId,
                                        TST_MasterPasswordAuthHandle,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPassword,
                                        TST_ATP_PassAuthObjectTestData[TST_index].TST_MasterPasswordSizeInBytes,
                                        TST_NO_CHALLENGE,
                                        TST_Workspace,
                                        TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationOpen - Secondary Authentication",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Open a transaction for operations */
    /*-----------------------------------*/
    TST_Status = SST_TransactionStart(&TST_TransactionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionStart",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Insert a new Data object */
    /*--------------------------*/
    TST_Status = SST_DataInsert(TST_TransactionId,
                                TST_SessionId, 
                                TST_MasterPasswordAuthHandle,
                                TST_ATP_PassAuthObjectTestData[TST_index].TST_Data,
                                TST_ATP_PassAuthObjectTestData[TST_index].TST_DataSize,
                                DX_TRUE,
                                TST_SimpleData,
                                &TST_DataHandle,
                                TST_Workspace,
                                TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataInsert",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Delete the data */
    /*-----------------*/
    TST_Status = SST_DataDelete(TST_TransactionId,
                                TST_SessionId,
                                TST_DataHandle,
                                TST_Workspace,
                                TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_DataDelete",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the master authentication */
    /*---------------------------------*/
    TST_Status = SST_AuthenticationClose(TST_SessionId,TST_MasterPasswordAuthHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticationClose",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Delete the authenticators */
    /*---------------------------*/
    TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                         TST_SecondaryPasswordAuthHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Secondary authenticator",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    TST_Status = SST_AuthenticatorDelete(TST_TransactionId,
                                         TST_MasterPasswordAuthHandle);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_AuthenticatorDelete - Master authenticator",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Close the transaction */
    /*-----------------------*/
    TST_Status = SST_TransactionEnd(TST_TransactionId,
                                    DX_TRUE,/*reflash sensitive*/
                                    TST_Workspace,
                                    TST_WorkspaceSizeInBytes);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_TransactionEnd",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


    /* Delete the session */
    /*--------------------*/
    TST_Status = SST_SessionDelete(TST_SessionId);

    TST_ATP_CHECK_ERROR_RETURN(TST_Status,"SST_SessionDelete",
                               "SEPQA_SST_PasswordAuthenticator",TST_MasterTestName);


  EXIT_ON_ERROR:
    return TST_Status;
}/* End of SST_TST_ATP_PasswordAuthenticator */



